# -*- coding: utf-8 -*-

import base64
import requests
import threading # Import threading for background task
import time # Import time for potential marquee effect (optional)
from PyQt5.QtCore import (
    Qt, pyqtSignal, QSettings,
    QUrl, QThread, QTimer, QPropertyAnimation, QRect,
    QEvent # <<< Import QEvent
)
from PyQt5.QtGui import QIcon, QDesktopServices, QFontMetrics
from PyQt5.QtWidgets import (
    QDialog, QLabel, QLineEdit,
    QVBoxLayout, QHBoxLayout, QFormLayout,
    QCheckBox, QPushButton, QDesktopWidget,
    QScrollArea, QSpacerItem, QSizePolicy # Import QScrollArea, QSpacerItem, QSizePolicy
)

from .config import API_BASE_URL, resource_path

# --- 修改：线程类名和信号名 ---
class SettingsFetcher(QThread):
    settings_ready = pyqtSignal(dict) # 信号传递字典

    def run(self):
        # 提供默认值
        settings_data = {
            'announcement': "公告加载失败",
            'window_title': "用户登录",
            'interface_title': "欢迎登录"
        }
        try:
            response = requests.post(
                f"{API_BASE_URL}",
                data={"action": "get_announcement"}, # API action 名称保持不变
                timeout=5
            )
            response.raise_for_status()
            result = response.json()
            if result.get("success") and "data" in result and isinstance(result["data"], dict):
                # 合并服务器数据和默认值，服务器优先
                server_data = result["data"]
                settings_data['announcement'] = server_data.get("announcement", settings_data['announcement'])
                settings_data['window_title'] = server_data.get("window_title", settings_data['window_title'])
                settings_data['interface_title'] = server_data.get("interface_title", settings_data['interface_title'])
            else:
                print(f"获取设置失败: {result.get('message', '未知错误')}")
        except Exception as e:
            print(f"获取设置时发生异常: {e}")

        self.settings_ready.emit(settings_data)
# --- 线程修改结束 ---


######################
# 登录窗口
######################
class LoginDialog(QDialog):
    login_success = pyqtSignal(dict)

    def __init__(self, software_name, software_version, parent=None):
        super().__init__(parent)
        self.software_name = software_name
        self.software_version = software_version
        self.announcement_animation = None # For marquee effect
        # 保存界面标题 QLabel 的引用
        self.title_label = None
        self.initUI()

        # 加载保存的账号信息
        self.loadSavedAccount()
        # 启动设置获取 (修改函数名)
        self.loadSettings()

    def initUI(self):
        # 不再在这里设置固定标题，由 updateDialogElements 设置
        # self.setWindowTitle("一可软件登录")
        # --- 暂时注释掉固定高度 ---
        # self.setFixedSize(400, 380)
        self.setFixedWidth(400) # 保持宽度固定，高度自适应
        self.setWindowIcon(QIcon(resource_path("icon.ico")))
        
        # 设置窗口样式 - 添加限定符以限制样式只应用于登录窗口组件
        self.setStyleSheet("""
            /* 基本窗口样式 */
            LoginDialog {
                background-color: #f8f9fa;
                font-family: 'Microsoft YaHei', sans-serif;
            }
            
            /* 标签样式 */
            LoginDialog QLabel {
                font-size: 14px;
                color: #333333;
            }
            
            /* 输入框样式 */
            LoginDialog QLineEdit {
                padding: 12px;
                border: 1px solid #ced4da;
                border-radius: 5px;
                font-size: 14px;
                background-color: white;
                min-height: 24px;
            }
            
            /* 按钮通用样式 */
            LoginDialog QPushButton {
                padding: 12px;
                border: none;
                border-radius: 5px;
                font-size: 14px;
                font-weight: bold;
            }
            
            /* 登录按钮样式 */
            LoginDialog QPushButton#loginButton {
                background-color: #007bff;
                color: white;
            }
            LoginDialog QPushButton#loginButton:hover {
                background-color: #0069d9;
            }
            LoginDialog QPushButton#loginButton:disabled {
                background-color: #6c757d;
            }
            
            /* 注册按钮样式 */
            LoginDialog QPushButton#registerButton {
                background-color: #28a745;
                color: white;
            }
            LoginDialog QPushButton#registerButton:hover {
                background-color: #218838;
            }
            
            /* 特殊标签样式 */
            LoginDialog #titleLabel {
                font-size: 24px;
                font-weight: bold;
                color: #007bff;
                margin-bottom: 20px;
            }
            LoginDialog #infoLabel {
                color: #6c757d;
                font-size: 12px;
            }
            
            /* 复选框样式 */
            LoginDialog QCheckBox {
                font-size: 13px;
                color: #555555;
            }
            
            /* 公告区样式 */
            LoginDialog QScrollArea#announcementArea {
                border: none;
                background-color: transparent; /* 背景透明 */
            }
            LoginDialog QLabel#announcementLabel {
                 color: #6c757d; /* 公告文字颜色 */
                 font-size: 12px;
                 background-color: transparent;
                 padding: 0 5px; /* 左右留一点边距 */
            }
        """)
        
        # 创建布局
        main_layout = QVBoxLayout()
        main_layout.setContentsMargins(30, 30, 30, 30)
        main_layout.setSpacing(15)
        
        # 标题 (创建但不设置文本，由 updateDialogElements 设置)
        self.title_label = QLabel("...") # 初始占位符
        self.title_label.setObjectName("titleLabel")
        self.title_label.setAlignment(Qt.AlignCenter)
        main_layout.addWidget(self.title_label)
        
        # --- 修改开始：使用 QHBoxLayout 替代 QFormLayout 实现手动对齐 ---
        # 用户名行
        username_layout = QHBoxLayout()
        username_label = QLabel("账号:")
        # username_label.setAlignment(Qt.AlignVCenter | Qt.AlignRight) # 在QHBoxLayout中，对齐通常由布局管理
        self.username_input = QLineEdit()
        self.username_input.setPlaceholderText("请输入账号")
        username_layout.addWidget(username_label)
        username_layout.addWidget(self.username_input)
        # 可以根据需要调整标签和输入框的伸展因子 username_layout.setStretchFactor(username_label, 1) ...
        main_layout.addLayout(username_layout)

        # 密码行
        password_layout = QHBoxLayout()
        password_label = QLabel("密码:")
        # password_label.setAlignment(Qt.AlignVCenter | Qt.AlignRight)
        self.password_input = QLineEdit()
        self.password_input.setPlaceholderText("请输入密码")
        self.password_input.setEchoMode(QLineEdit.Password)
        password_layout.addWidget(password_label)
        password_layout.addWidget(self.password_input)
        main_layout.addLayout(password_layout)
        # --- 修改结束 ---
        
        # 记住密码
        self.remember_checkbox = QCheckBox("记住我的账号")
        main_layout.addWidget(self.remember_checkbox)
        
        # 按钮区域
        buttons_layout = QVBoxLayout()
        buttons_layout.setSpacing(10)
        
        # 登录按钮
        self.login_button = QPushButton("登录")
        self.login_button.setObjectName("loginButton")
        self.login_button.setMinimumHeight(40)
        self.login_button.clicked.connect(self.handleLogin)
        buttons_layout.addWidget(self.login_button)
        
        # 注册按钮
        register_button = QPushButton("注册账号")
        register_button.setObjectName("registerButton")
        register_button.setMinimumHeight(40)
        register_button.clicked.connect(self.openRegisterPage)
        buttons_layout.addWidget(register_button)
        
        main_layout.addLayout(buttons_layout)

        # --- 修改：移动公告区域到布局底部 ---
        # 旧的信息标签 (已移除)

        # 错误提示标签 (先添加)
        self.error_label = QLabel("")
        self.error_label.setStyleSheet("color: #dc3545; font-size: 13px;")
        self.error_label.setAlignment(Qt.AlignCenter)
        self.error_label.hide()
        main_layout.addWidget(self.error_label)

        # 新增公告滚动区域 (后添加)
        self.announcement_scroll_area = QScrollArea()
        self.announcement_scroll_area.setObjectName("announcementArea")
        self.announcement_scroll_area.setWidgetResizable(True)
        self.announcement_scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff) # 禁用水平滚动条 (靠动画)
        self.announcement_scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)   # 禁用垂直滚动条
        self.announcement_scroll_area.setFixedHeight(25) # 设置一个合适的高度

        # >>> 启用鼠标跟踪 <<<
        self.announcement_scroll_area.setMouseTracking(True)
        # >>> 安装事件过滤器 (在 self 上处理) <<<
        self.announcement_scroll_area.installEventFilter(self)

        self.announcement_label = QLabel(" 加载中...") # 初始文本
        self.announcement_label.setObjectName("announcementLabel")
        self.announcement_label.setAlignment(Qt.AlignVCenter | Qt.AlignLeft) # 垂直居中，左对齐

        self.announcement_scroll_area.setWidget(self.announcement_label)
        main_layout.addWidget(self.announcement_scroll_area)
        # --- 修改结束 ---

        self.setLayout(main_layout)
        
        # 居中显示
        self.center()
    
    def center(self):
        """将窗口居中显示在屏幕上"""
        qr = self.frameGeometry()
        cp = QDesktopWidget().availableGeometry().center()
        qr.moveCenter(cp)
        self.move(qr.topLeft())
    
    def loadSavedAccount(self):
        """加载已保存的账号信息"""
        settings = QSettings("YikeAIGC", "YikeSoftware")
        remember = settings.value("remember", False, type=bool)
        
        # 只有在选择了"记住我的账号"时才自动填充用户名和密码
        if remember:
            username = settings.value("username", "")
            password = settings.value("password", "")
            
            if username:
                self.username_input.setText(username)
            
            if password:
                # 解密密码
                try:
                    password = self.decryptPassword(password)
                    self.password_input.setText(password)
                    self.remember_checkbox.setChecked(True)
                except:
                    pass
    
    def saveAccount(self, username, password, remember):
        """保存账号信息"""
        settings = QSettings("YikeAIGC", "YikeSoftware")
        settings.setValue("username", username)
        
        if remember:
            # 加密密码
            encrypted_password = self.encryptPassword(password)
            settings.setValue("password", encrypted_password)
        else:
            settings.setValue("password", "")
        
        settings.setValue("remember", remember)
        # 新增一个设置项，明确表示不自动登录
        settings.setValue("auto_login", False)
    
    def encryptPassword(self, password):
        """简单加密密码"""
        key = "YikeAIGC_2023"
        encrypted = []
        for i, c in enumerate(password):
            key_c = key[i % len(key)]
            encrypted_c = chr(ord(c) + ord(key_c) % 256)
            encrypted.append(encrypted_c)
        return base64.b64encode("".join(encrypted).encode()).decode()
    
    def decryptPassword(self, encrypted):
        """解密密码"""
        key = "YikeAIGC_2023"
        encrypted = base64.b64decode(encrypted).decode()
        decrypted = []
        for i, c in enumerate(encrypted):
            key_c = key[i % len(key)]
            decrypted_c = chr(ord(c) - ord(key_c) % 256)
            decrypted.append(decrypted_c)
        return "".join(decrypted)
    
    def showError(self, message):
        """显示错误消息"""
        self.error_label.setText(message)
        self.error_label.show()
    
    def hideError(self):
        """隐藏错误消息"""
        self.error_label.hide()
    
    # --- 修改：加载和更新函数名，并更新所有元素 --- 
    def loadSettings(self):
        self.fetcher = SettingsFetcher(self) # 使用新类名
        self.fetcher.settings_ready.connect(self.updateDialogElements) # 连接到新槽函数
        self.fetcher.start()

    def updateDialogElements(self, settings_data):
        # 更新窗口标题
        self.setWindowTitle(settings_data.get('window_title', '用户登录'))
        # 更新界面大标题
        if self.title_label:
            self.title_label.setText(settings_data.get('interface_title', '欢迎登录'))
        # 更新公告文本
        self.announcement_label.setText(settings_data.get('announcement', ''))
        self.announcement_label.adjustSize()
        # 启动公告滚动动画
        self.startMarqueeAnimation()
    # --- 修改结束 ---

    # --- 新增：跑马灯效果 --- 
    def startMarqueeAnimation(self):
        # 停止之前的动画
        if self.announcement_animation and self.announcement_animation.state() == QPropertyAnimation.Running:
            self.announcement_animation.stop()

        # 计算文本宽度和容器宽度
        label_width = self.announcement_label.fontMetrics().boundingRect(self.announcement_label.text()).width()
        scroll_area_width = self.announcement_scroll_area.width() - 10 # 减去一些边距

        # 只有当文本宽度超过容器宽度时才启动动画
        if label_width > scroll_area_width:
             # 让 Label 宽度适应文本，这样动画才有意义
            self.announcement_label.setFixedWidth(label_width + 20) # 加一点额外空间

            # 初始位置 (从右侧边缘开始)
            start_x = scroll_area_width
            # 结束位置 (完全移出左侧边缘)
            end_x = -label_width

            # 创建动画
            self.announcement_animation = QPropertyAnimation(self.announcement_label, b"geometry", self)
            duration = int(label_width * 15) # 根据文本长度调整滚动速度 (毫秒)
            self.announcement_animation.setDuration(duration)

            # 设置动画路径
            start_rect = QRect(start_x, 0, self.announcement_label.width(), self.announcement_label.height())
            end_rect = QRect(end_x, 0, self.announcement_label.width(), self.announcement_label.height())
            self.announcement_animation.setStartValue(start_rect)
            self.announcement_animation.setEndValue(end_rect)

            # 无限循环
            self.announcement_animation.setLoopCount(-1)
            self.announcement_animation.start()
        else:
            # 如果文本不超长，确保Label宽度不超过ScrollArea，并居中（或左对齐）
            self.announcement_label.setFixedWidth(scroll_area_width)
            # self.announcement_label.setAlignment(Qt.AlignCenter) # 如果想居中显示短公告
            pass # 不需要动画

    # 重写 resizeEvent 以便在窗口大小变化时重新计算动画
    def resizeEvent(self, event):
        super().resizeEvent(event)
        # 延迟一点执行，确保布局更新完毕
        QTimer.singleShot(50, self.startMarqueeAnimation)
    # --- 跑马灯效果结束 ---

    # >>> 新增：事件过滤器方法 <<<
    def eventFilter(self, obj, event):
        # 检查事件是否发生在公告区域
        if obj is self.announcement_scroll_area:
            if event.type() == QEvent.Enter:
                # 鼠标进入: 暂停动画
                if self.announcement_animation and self.announcement_animation.state() == QPropertyAnimation.Running:
                    self.announcement_animation.pause()
                    # print("Animation Paused") # for debugging
                return True # 事件已处理
            elif event.type() == QEvent.Leave:
                # 鼠标离开: 恢复动画
                if self.announcement_animation and self.announcement_animation.state() == QPropertyAnimation.Paused:
                    self.announcement_animation.resume()
                    # print("Animation Resumed") # for debugging
                return True # 事件已处理

        # 对于其他对象或事件，调用基类的 eventFilter
        return super().eventFilter(obj, event)
    # >>> 事件过滤器结束 <<<

    def handleLogin(self):
        """处理登录请求"""
        self.hideError()
        
        username = self.username_input.text().strip()
        password = self.password_input.text().strip()
        
        if not username:
            self.showError("请输入账号")
            return
        
        if not password:
            self.showError("请输入密码")
            return
        
        remember = self.remember_checkbox.isChecked()
        
        # 禁用登录按钮
        self.login_button.setEnabled(False)
        self.login_button.setText("登录中...")
        
        try:
            response = requests.post(
                f"{API_BASE_URL}",
                data={
                    "action": "login",
                    "username": username,
                    "password": password,
                    "software": self.software_name
                },
                timeout=10
            )
            
            # 解析响应
            data = response.json()
            
            if data["success"]:
                # 保存账号信息
                self.saveAccount(username, password, remember)
                
                # 安全地获取数据字段，避免缺少字段导致错误
                server_data = data.get("data", {})
                
                # 打印服务器返回的完整信息以便调试
                print(f"服务器返回数据: {server_data}")
                
                # 获取用户角色 - 先尝试user_role字段，如果不存在再尝试role字段，最后尝试JWT令牌解析
                user_role = server_data.get("user_role", "")
                if not user_role:
                    user_role = server_data.get("role", "")
                    
                # 如果服务器角色为空，但JWT令牌中包含角色信息，则尝试从令牌中提取
                if not user_role and "role" in server_data.get("token", ""):
                    import jwt
                    try:
                        token = server_data.get("token", "")
                        token_data = jwt.decode(token, options={"verify_signature": False})
                        user_role = token_data.get("role", "")
                        print(f"从令牌中提取角色: {user_role}")
                    except Exception as e:
                        print(f"令牌解析失败: {str(e)}")
                
                # 发送登录成功信号，使用统一的字段名role
                user_info = {
                    "user_id": server_data.get("user_id", ""),
                    "username": server_data.get("username", username),
                    "token": server_data.get("token", ""),
                    "role": user_role,  # 使用正确提取的角色
                    "vip_expire_date": server_data.get("vip_expire_date", "")
                }
                
                # 打印调试信息
                print(f"登录成功，用户信息: {user_info}")
                
                self.login_success.emit(user_info)
                self.accept()  # 关闭对话框，返回Accepted
            else:
                # 显示错误消息
                self.showError(data.get("message", "登录失败，请重试"))
                self.login_button.setEnabled(True)
                self.login_button.setText("登录")
        except Exception as e:
            error_message = f"网络错误: {str(e)}"
            print(f"登录异常: {error_message}")
            self.showError(error_message)
            self.login_button.setEnabled(True)
            self.login_button.setText("登录")
    
    def openRegisterPage(self):
        """打开注册页面"""
        QDesktopServices.openUrl(QUrl("https://tools.yikeaigc.com/sys/index.php")) 