# -*- coding: utf-8 -*-

import requests
from PyQt5.QtWidgets import QMessageBox
from PyQt5.QtCore import QSettings, QUrl
from PyQt5.QtGui import QDesktopServices
import json # 确保导入 json
import webbrowser # 导入 webbrowser 作为备选打开方式

from .config import API_BASE_URL

######################
# Token管理类
######################
class TokenManager:
    def __init__(self):
        self.token = None
        self.user_id = None
        self.username = None
        self.role = None
        self.vip_expire_date = None
    
    def setUserInfo(self, user_info):
        """设置用户信息"""
        # 使用get方法安全地获取用户信息
        self.token = user_info.get("token")
        self.user_id = user_info.get("user_id")
        self.username = user_info.get("username")
        self.role = user_info.get("role")
        self.vip_expire_date = user_info.get("vip_expire_date")
    
    def _make_api_request(self, action, data=None, timeout=10):
        """
        统一处理对后端 API 的请求。

        Args:
            action (str): 请求的操作标识。
            data (dict, optional): 发送的 POST 数据 (除了 action 和 token)。Defaults to None.
            timeout (int, optional): 请求超时时间。Defaults to 10.

        Returns:
            tuple: (bool, dict or str): 第一个元素表示请求是否逻辑成功 (HTTP 成功且 JSON 解析成功)，
                     第二个元素是成功时服务器返回的 **完整 JSON 响应字典** 或失败时的错误消息 (str)。
        """
        if not self.token and action not in ['login', 'register']: # 登录注册不需要token
             return False, "用户未登录"

        headers = {}
        post_data = data.copy() if data else {}
        post_data['action'] = action

        if self.token:
            headers['Authorization'] = f'Bearer {self.token}'
            # 也将 token 加入 post_data 作为备选，以兼容旧的或不支持 header 的服务器配置
            post_data['token'] = self.token

        try:
            response = requests.post(
                f"{API_BASE_URL}",
                headers=headers,
                data=post_data,
                timeout=timeout
            )
            response.raise_for_status() # 检查 HTTP 错误 (4xx, 5xx)

            try:
                result = response.json()
                # 直接返回解析后的完整 JSON 字典
                return True, result
            except json.JSONDecodeError:
                return False, f"无法解析服务器响应: {response.text}"
        except requests.exceptions.RequestException as e:
            error_message = f"网络错误或服务器无响应: {str(e)}"
            print(f"API 请求异常 ({action}): {error_message}")
            return False, error_message
        except Exception as e:
            # 其他意外错误
            error_message = f"处理API请求时发生未知错误: {str(e)}"
            print(f"API 请求未知异常 ({action}): {error_message}")
            return False, error_message

    def verifyToken(self):
        """验证令牌有效性，只检查令牌本身是否有效，不检查软件访问权限"""
        if not self.token:
            return False
        
        success, result = self._make_api_request("verify_token")
        
        if success and result.get("success"): # 检查 API 逻辑成功
            # 从返回的 data 字段获取用户信息
            response_data = result.get("data", {})
            self.user_id = response_data.get("user_id", self.user_id)
            self.username = response_data.get("username", self.username)
            self.role = response_data.get("role", self.role) # verify_token 返回的是 'role'
            self.vip_expire_date = response_data.get("vip_expire_date", self.vip_expire_date)
            return True
        else:
            # result 在 success=True 时是字典，在 success=False 时是字符串
            error_message = result.get("message", "未知错误") if isinstance(result, dict) else result
            print(f"验证令牌失败: {error_message}")
            return False
    
    def verify_permission_with_server(self, software_name):
        """
        [新] 调用服务器端点检查用户对特定软件的访问权限。

        Args:
            software_name (str): 需要检查权限的软件名称。

        Returns:
            tuple: (bool, str): 第一个元素表示是否允许访问，第二个元素是服务器返回的消息。
        """
        if not self.token:
            return False, "用户未登录"
        if not software_name:
            return False, "未提供软件名称"

        try:
            success, result = self._make_api_request(
                "check_permission",
                data={"software_name": software_name}
            )

            # 检查软件是否存在
            if success and isinstance(result, dict):
                # 从顶层获取 message
                message = result.get("message", "未知响应")
                
                # 特定处理：如果php返回success=false，可能是软件不存在
                if not result.get("success", False):
                    return False, "程序已下架，请联系管理员微信wxcwxc1989"
                
                # 如果data为空或不是字典，可能是软件不存在
                data_part = result.get("data")
                if data_part is None or not isinstance(data_part, dict):
                    return False, "程序已下架，请联系管理员微信wxcwxc1989"
                
                # 从 data 子字典获取 allowed 状态
                allowed = data_part.get("allowed", False)
                return allowed, message
            else:
                # API请求失败时返回自定义错误消息
                return False, "程序已下架，请联系管理员微信wxcwxc1989"
        except Exception as e:
            # 遇到任何异常，返回自定义错误消息
            print(f"验证权限时发生错误: {str(e)}")
            return False, "程序已下架，请联系管理员微信wxcwxc1989"

    def checkAccess(self, software_name):
        """
        检查软件访问权限 (现在调用新的服务器端验证)。
        如果访问被拒绝，会弹出消息框。
        对于VIP权限不足的情况，会提示升级并尝试打开用户中心。

        Args:
            software_name (str): 需要检查权限的软件名称。

        Returns:
            bool: 是否允许访问。
        """
        allowed, message = self.verify_permission_with_server(software_name)

        if not allowed:
            print(f"DEBUG: Server denied access. Raw message received: '{message}'")
            print(f"访问被拒绝: {software_name}, 原因: {message}")
            # 检查是否是 VIP 相关的拒绝信息
            is_vip_issue = False
            vip_keywords = ["VIP权限", "VIP已过期", "需要VIP"]
            for keyword in vip_keywords:
                if keyword in message:
                    is_vip_issue = True
                    break

            if is_vip_issue:
                # 显示自定义的 VIP 提示并尝试打开链接
                custom_message = '此工具需要VIP账号，请升级账号。\n\n点击"确定"将尝试打开用户中心。'
                reply = QMessageBox.warning(None, "需要升级", custom_message, QMessageBox.Ok | QMessageBox.Cancel)
                if reply == QMessageBox.Ok:
                    try:
                        # 优先使用 PyQt5 的方式打开链接
                        url = QUrl("https://tools.yikeaigc.com/sys/index.php")
                        if not QDesktopServices.openUrl(url):
                             # 如果 PyQt5 方式失败，尝试使用 webbrowser
                             print("QDesktopServices failed, trying webbrowser...")
                             webbrowser.open("https://tools.yikeaigc.com/sys/index.php")
                    except Exception as e:
                         print(f"打开URL时出错: {e}")
                         # 即使打开失败，也告知用户需要升级
                         QMessageBox.information(None, "提示", "无法自动打开浏览器，请手动访问用户中心升级账号。")
            else:
                # 对于非 VIP 问题，显示原始错误信息
                QMessageBox.warning(None, "访问受限", f"{message}")

        return allowed

    def saveToken(self):
        """
        保存令牌到设置
        注：已禁用自动保存功能，确保每次启动程序都需要重新登录
        """
        # 注释掉保存令牌的代码，确保每次启动都需要登录
        # if self.token:
        #     settings = QSettings("YikeAIGC", "YikeSoftware")
        #     settings.setValue("token", self.token)
        pass
    
    def loadToken(self):
        """从设置中加载令牌"""
        # 禁用自动登录功能
        return False
        
        # 下面的代码不会执行
        # settings = QSettings("YikeAIGC", "YikeSoftware")
        # saved_token = settings.value("token", "")
        # if saved_token:
        #     self.token = saved_token
        #     return True
        # return False

# 创建全局TokenManager
token_manager = TokenManager() 