# -*- coding: utf-8 -*-
"""
认证模块集成辅助工具
用于将auth_module认证模块集成到任何程序中，同时保持程序原有UI风格
"""

import sys
import jwt
from auth_module import LoginDialog, token_manager, API_BASE_URL
from PyQt5.QtWidgets import QMessageBox, QDialog
from PyQt5.QtCore import QUrl # 确保导入 QUrl
from PyQt5.QtGui import QDesktopServices # 确保导入 QDesktopServices
import requests
import json
import webbrowser # 导入 webbrowser 作为备选打开方式

def integrate_auth(software_name, vip_only=False):
    """
    集成认证模块到应用程序
    
    参数:
    software_name (str): 软件名称（必须与数据库中注册的名称匹配）
    vip_only (bool): 是否仅限VIP用户使用（此参数现在主要用于客户端初步判断，最终决策在服务器）
    
    返回:
    bool: 认证是否成功且有权限访问
    dict: 用户信息 (如果认证成功)
    """
    try:
        print(f"开始认证过程，软件名称: {software_name}, 要求VIP: {vip_only}")
        
        # 创建登录对话框，保持其原有风格
        login_dialog = LoginDialog(software_name, "")  # 不传版本号
        
        # 用于接收登录成功信号
        user_info = None
        
        def on_login_success(info):
            nonlocal user_info
            user_info = info
            print(f"登录成功回调，接收到用户信息: {info}")
            
            # 设置 token manager 中的用户信息 (由 LoginDialog 返回)
            token_manager.setUserInfo(info)
        
        # 连接登录成功信号
        login_dialog.login_success.connect(on_login_success)
        
        # 显示登录对话框
        result = login_dialog.exec_()
        
        # 用户取消登录或登录对话框内部验证失败 (如网络错误)
        if result != QDialog.Accepted or not user_info or not token_manager.token:
            print("登录过程未完成或用户取消")
            # LoginDialog 内部应该已经提示了具体错误
            return False, None
        
        # --- 核心权限验证：调用服务器 --- 
        print(f"登录成功，开始调用服务器检查权限: {software_name}")
        # 使用 token_manager 中新的服务器验证方法
        try:
            allowed, message = token_manager.verify_permission_with_server(software_name)
        except Exception as e:
            print(f"认证集成过程发生意外错误: {str(e)}")
            QMessageBox.critical(None, "认证失败", "程序已下架，请联系管理员微信wxcwxc1989")
            return False, user_info

        if not allowed:
            print(f"服务器权限验证失败: {message}")
            # 检查是否有自定义消息
            if "程序已下架" in message:
                QMessageBox.critical(None, "程序已下架", message)
                return False, user_info
                
            # 检查是否是 VIP 相关的拒绝信息
            is_vip_issue = False
            vip_keywords = ["VIP权限", "VIP已过期", "需要VIP"]
            for keyword in vip_keywords:
                if keyword in message:
                    is_vip_issue = True
                    break

            if is_vip_issue:
                # 显示自定义的 VIP 提示并尝试打开链接
                custom_message = '此工具需要VIP账号，请升级账号。\n\n点击"确定"将尝试打开用户中心。'
                reply = QMessageBox.critical(None, "需要升级", custom_message, QMessageBox.Ok | QMessageBox.Cancel)
                if reply == QMessageBox.Ok:
                    try:
                        # 优先使用 PyQt5 的方式打开链接
                        url = QUrl("https://tools.yikeaigc.com/sys/index.php")
                        if not QDesktopServices.openUrl(url):
                            # 如果 PyQt5 方式失败，尝试使用 webbrowser
                            print("QDesktopServices failed, trying webbrowser...")
                            webbrowser.open("https://tools.yikeaigc.com/sys/index.php")
                    except Exception as e:
                         print(f"打开URL时出错: {e}")
                         # 即使打开失败，也告知用户需要升级
                         QMessageBox.information(None, "提示", "无法自动打开浏览器，请手动访问用户中心升级账号。")
            else:
                 # 对于非 VIP 问题，显示原始错误信息
                 QMessageBox.critical(None, "访问受限", message)

            # 即使登录成功，如果没有权限，也返回 False
            # 返回 user_info 允许调用者知道是哪个用户被拒绝
            return False, user_info
        else:
            print("服务器权限验证成功")
            # 如果需要，可以在这里根据 vip_only 和 user_info['role'] 做一个额外的客户端检查
            # 但服务器的判断是最终依据
            if vip_only:
                 # 从 token_manager 获取最新的角色信息（可能在 verify_permission_with_server 中更新了）
                 current_role = (token_manager.role or '').lower()
                 if current_role not in ['vip', 'admin']:
                     # 这种情况理论上不应发生，因为服务器已经判断 allowed=True
                     # 但作为双重检查，如果服务器逻辑有误，这里可以捕获
                     print(f"警告：服务器允许访问，但客户端检查VIP权限失败 (角色: {current_role})")
                     # 可以选择在这里也拒绝访问
                     # QMessageBox.warning(None, "权限警告", "用户角色与软件要求不符，请联系管理员。")
                     # return False, user_info

            # 认证和授权均成功
            print("认证与授权成功")
            # 返回最新的用户信息（可能已通过 API 调用更新）
            updated_user_info = {
                "token": token_manager.token,
                "user_id": token_manager.user_id,
                "username": token_manager.username,
                "role": token_manager.role,
                "vip_expire_date": token_manager.vip_expire_date
            }
            return True, updated_user_info
    
    except Exception as e:
        # 处理意外异常
        error_message = f"认证集成过程发生意外错误: {str(e)}"
        print(error_message)
        import traceback
        traceback.print_exc()
        QMessageBox.critical(None, "认证错误", error_message)
        return False, None 